// player.h: Beispielspieler fr Asteroids
// Harald Bgeholz / c't
#if defined(_WINDOWS)
#define ADDRESS DWORD
#else
#define SOCKET int
#define ADDRESS in_addr_t
// 3 Includes fr sockaddr_in
#include <sys/types.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#endif
#include "math.h"

inline void assert(bool b)
{
	if(b)
		return;
	printf("Assertion fehlgeschlagen!\n");
	while(true)
		;
}

static const int MAX_ASTEROIDS = 100;
static const int MAX_SHOTS = 10;
typedef unsigned char byte;
typedef unsigned short word;
typedef unsigned int dword;
const int MaxInt = 0x7FFFFFFF;

//#pragma warning( disable: 4309)

class Asteroid
{
public:
	int x;    // Koordinaten des Mittelpunkts
	int y;
	int type; // 1 ... 4, uere Form
	int sf;   // scale factor: 0 = gro, 15 = mittel, 14 = klein

	void set(int x, int y, int type, int sf);
};

class Shot
{
public:
	int x;
	int y;

	void set(int x, int y);
};

class GameStatus
{
public:
	bool ship_present;  // Schiff sichtbar
	int ship_x;         // Mittelpunkt des Schiffs
	int ship_y;
	int ship_dx;        // Blickrichtung des Schiffes
	int ship_dy;
	bool saucer_present;// UFO sichtbar
	int saucer_x;       // Mittelpunkt des UFOs
	int saucer_y;
	int saucer_size;    // Gre: 15 = gro, 14 = klein
	int nasteroids; // Anzahl Asteroiden
	Asteroid asteroids[MAX_ASTEROIDS];
	int nshots;     // Anzahl Schsse
	Shot shots[MAX_SHOTS];
	void clear(void);
};

#pragma pack(1)

class KeysPacket
{
private:
	char signature[6];
public:
	static const char KEY_HYPERSPACE = 1;
	static const char KEY_FIRE = 2;
	static const char KEY_THRUST = 4;
	static const char KEY_RIGHT = 8;
	static const char KEY_LEFT = 0x10;

	char keys;
	char ping;     // wird vom Server bei nchster Gelegenheit zurckgeschickt. Fr Latenzmessung.

	KeysPacket(void);
	void clear(void);         // alle Tasten loslassen
	void hyperspace(bool b);  // Hyperspace drcken (true) oder loslassen (false)
	void fire(bool b);        // Feuerknopf drcken (true) oder loslassen (false)
	void thrust(bool b);      // Beschleunigen ...
	void right(bool b);       // rechts drehen ...
	void left(bool b);        // links drehen
};


#if 0
enum RAM_Type
{
/*0x18*/	RAM_CurrentPlayer,
/*0x1C*/	RAM_NumPlayers,
/*0x52*/	RAM_PlayerScoreTens,
/*0x53*/	RAM_PlayerScoreThousends,
/*0x57*/	RAM_cShips,
/*0x59*/	RAM_HyperSpaceStatus,
/*0x5C*/	RAM_FastTimer,
/*0x5D*/	RAM_SlowTimer,
/*0x5F*/	RAM_RandomSeed1,
/*0x60*/	RAM_RandomSeed2,
/*0x61*/	RAM_ShipDirection,
/*0x63*/	RAM_FireHyperspaceToggle,
/*0x64*/	RAM_AccX,
/*0x65*/	RAM_AccY,
/*0x02F5*/	RAM_NumAsteroidsStart,
/*0x02F6*/	RAM_NumAsteroids,
/*0x02F7*/	RAM_SaucerCountdown,
/*0x02F8*/	RAM_SaucerCountdownStart,
/*0x02FA*/	RAM_InvisibleTimer,
/*0x02FB*/	RAM_NewLevelStartCountdown,
///*0x02FC*/	RAM_2FC,
///*0x02FD*/	RAM_2FD,
			RAM_Count
};
const int ByteAddress[RAM_Count] = 
{
	0x18,
	0x1C,
	0x52,
	0x53,
	0x57,
	0x59,
	0x5C,
	0x5D,
	0x5F,
	0x60,
	0x61,
	0x63,
	0x64,
	0x65,
	0x02F5,
	0x02F6,
	0x02F7,
	0x02F8,
	0x02FA,
	0x02FB,
//	0x02FC,
//	0x02FD,
};

enum RAMO_Type
{
/*0x200*/	RAMO_Status,
/*0x223*/	RAMO_VelocityX,
/*0x246*/	RAMO_VelocityY,
/*0x269*/	RAMO_PositionX,
/*0x28C*/	RAMO_PositionY,
/*0x2AF*/	RAMO_PositionExactX,
/*0x2D2*/	RAMO_PositionExactY,
			RAMO_Count
};

const int RAMO_Address[RAMO_Count] =
{
	0x200,
	0x223,
	0x246,
	0x269,
	0x28C,
	0x2AF,
	0x2D2,
};
#endif

const int cAsteroids = 27;
const int cPlayerShots = 4;
const int cSaucerShots = 2;

enum
{
	AsteroidMinIndex,
	AsteroidMaxIndex = AsteroidMinIndex+cAsteroids-1,
	PlayerIndex,
	SaucerIndex,
	SaucerShotMinIndex,
	SaucerShotMaxIndex = SaucerShotMinIndex+cSaucerShots-1,
	PlayerShotMinIndex,
	PlayerShotMaxIndex = PlayerShotMinIndex+cPlayerShots-1,
	NumObjects
};

const int ShotLifetime = 0x12 * 4;
const int RotationSpeed = 3;

inline bool MoveLinearXSingle(word& PositionX, char VelocityX)
{
	PositionX += VelocityX;
	if(PositionX >= 0x2000)//Objekt geht rechts oder links raus
	{
		PositionX &= 0x1FFF;//andere Seite wieder rein
		return true;
	}
	return false;
}

inline word MoveLinearX(word PositionX, char VelocityX, int Steps)
{
	return (PositionX + VelocityX*Steps) & 0x1FFF;
/*	word oldPositionX = PositionX;
	for(int i = 0; i < Steps; i++)
	{
		PositionX += char(VelocityX);
		if(PositionX >= 0x2000)//Objekt geht rechts raus
		{
			PositionX &= 0x1FFF;//links wieder rein
		}
	}
	if(PositionX != ((oldPositionX+char(VelocityX)*Steps)&0x1FFF))
		assert(0);
	return PositionX;*/
}

inline word MoveLinearY(word PositionY, char VelocityY, int Steps)
{
	PositionY += VelocityY*Steps;
	if(VelocityY >= 0)
	{
		if(PositionY >= 0x1800)
			PositionY -= 0x1800;
	}
	else
		if(PositionY >= 0x1800)
			PositionY += 0x1800;
	return PositionY;
/*	word oldPositionY = PositionY;
	word realoldPositionY = PositionY;
	for(int i = 0; i < Steps; i++)
	{
		PositionY += char(VelocityY);
		if(PositionY>>8 == 0x0018)//Objekt geht unten raus
		{
			PositionY = 0x0000 | (PositionY&0x00FF);
		}
		else if(PositionY >= 0x1800)
		{
			PositionY = 0x1700 | (PositionY&0x00FF);
		}
	}
	if(PositionY != oldPositionY)
		assert(0);
	return PositionY;*/
}


struct DirInfo
{
	word StartX, StartY;
	char VelocityX, VelocityY;
	int CollisionAtTime;
	int ShotTime;

	int PositionX(int t)
	{
		return MoveLinearX(StartX, VelocityX, t);
	}
	int PositionY(int t)
	{
		return MoveLinearY(StartY, VelocityY, t);
	}
	int PositionXEasy(int t)
	{
		return int(StartX)+int(VelocityX)*t;
	}
	int PositionYEasy(int t)
	{
		return int(StartY)+int(VelocityY)*t;
	}
};
extern DirInfo BestDirInfo[2];
extern bool ErrorsOccured;
extern bool updating;
extern __int64* pScore;

inline word MakeWord(byte high, byte low)
{
	return word(high)<<8 | word(low);
}
inline void MakeBytesFromWord(word Word, byte& High, byte& Low)
{
	High = byte(Word>>8);
	Low = byte(Word);
}
inline byte& LowByte(word& Word)
{
	return (byte&)Word;
}
inline byte& HighByte(word& Word)
{
	return *(((byte*)&Word)+1);
}
inline byte Half(byte value)
{
	if(char(value) >= 0)
		return value/2;
	else
		return char(value-1)/2;
}
inline void Clamp(byte& value, byte min, byte max)
{
	if(char(value) >= 0)
	{
		if(value > max)
			value = max;
	}
	else
	{
		if(value < min)
			value = min;
	}
}
template<typename T>
T Min(T a, T b)
{
	return a<b ? a : b;
}
inline word DecimalToBin(byte High, byte Low)
{
	return (High>>4)*1000 + (High&0x0F)*100 + (Low>>4)*10 + (Low&0x0F);
}
inline void BinToDecimal(word Bin, byte& High, byte& Low)
{
	Low = Bin%10;
	Bin /= 10;
	Low |= Bin%10 << 4;
	Bin /= 10;
	High = Bin%10;
	Bin /= 10;
	High |= Bin%10 << 4;
}
inline byte& ShiftLeft(byte& byte, bool& carry)
{
	carry = (byte>>7) != 0;
	byte <<= 1;
	return byte;
}
inline byte& RotateLeft(byte& byte, bool& carry)
{
	bool newcarry = (byte>>7) != 0;
	byte <<= 1;
	if(carry) byte |= 1;
	carry = newcarry;
	return byte;
}
inline byte& ShiftRight(byte& byte, bool& carry)
{
	carry = (byte&1) != 0;
	byte >>= 1;
	return byte;
}
inline byte& RotateRight(byte& byte, bool& carry)
{
	bool newcarry = (byte&1) != 0;
	byte >>= 1;
	if(carry) byte |= 0x80;
	carry = newcarry;
	return byte;
}
inline byte& Add(byte& a, byte b, bool& carry)
{
	int sum = a + int(b) + carry;
	carry = (sum & 0xFF00) != 0;
	a = (byte)sum;
	return a;
}
inline byte& Subtract(byte& a, byte b, bool& carry)
{
	int c = (carry ? 1 : 0) ^ 1;
	int dif = a - int(b) - c;
	carry = (dif & 0xff00) == 0;
	a = (byte)dif;
	return a;
}

class Object
{
public:
	char Status;
	word PositionX, PositionY;
	char VelocityX, VelocityY;
};

class ShotInfo
{
public:
	ShotInfo()
	{
		Index = -1;
		StartTime = ShootTime = -1;
		CollisionTime = -1;
	}
public:
	int Index;
	word PositionX, PositionY;
	int StartTime, ShootTime;
	int CollisionTime;
	int Target;
	word StartX, StartY;
	char VelocityX, VelocityY;
};

class GameRam
{
	byte CurrentPlayer;         
	byte NumPlayers;
	byte PlayerScoreTens, PlayerScoreThousends;	
	byte cShips;				
	char HyperSpaceStatus;		
	byte FastTimer, SlowTimer;				
	byte RandomSeed1, RandomSeed2;			
	byte ShipDirection;			
	byte FireHyperspaceToggle;	
	byte AccX;					
	byte AccY;					
	byte NumAsteroidsStart;     
	byte NumAsteroids;			
	byte SaucerCountdown;		
	byte SaucerCountdownStart;	
	byte InvisibleTimer;		
	byte NewLevelStartCountdown;

	ShotInfo ShotInfos[cPlayerShots];

public:
	Object Objects[NumObjects];
public:
	void StartNewGame()
	{
		memset(this, 0, sizeof(*this));
		NumPlayers = 1;
		cShips = 3;
		RandomSeed1 = 0;
		NumAsteroidsStart = 2;
		InvisibleTimer = 1;
		SaucerCountdownStart = SaucerCountdown = 0x92;
		NewLevelStartCountdown = 0x7F;
		ResetPlayerPosition();
		for(int i = 0; i < cPlayerShots; i++)
			ShotInfos[i] = ShotInfo();
	}
	const Object* GetObjects() const
	{
		return Objects;
	}
	KeysPacket CalcDirInfos(ShotInfo* pShotInfo, int* pDeltaDir, bool CalcSecondShoot);
	void Render(HDC hdcScreen, int cx, int cy)
	{
		HBITMAP hBitmap = CreateCompatibleBitmap(hdcScreen, cx, cy);
		HDC hdc = CreateCompatibleDC(hdcScreen);
		SelectObject(hdc, hBitmap);

		RECT rc;
		SetRect(&rc, 0, 0, cx, cy);
		FillRect(hdc, &rc, (HBRUSH)GetStockObject(updating ? WHITE_BRUSH : LTGRAY_BRUSH));


		SelectObject(hdc, GetStockObject(NULL_BRUSH));
		for(int Pass = 0; Pass < 2; Pass++)
		{
			for(int i = 0; i < NumObjects; i++)
			{
				if(Objects[i].Status==0)
					continue;
				if(Objects[i].Status < 0 && Pass != 0 || //Explosion im ersten Pass zeichnen
					Objects[i].Status >= 0 && Pass == 0)
					continue;
				int Size = 100;
				if(Objects[i].Status < 0)//Explosion
					Size = -Objects[i].Status;
				else if(i==PlayerIndex)
					Size = 12;
				else if(i==SaucerIndex)
					Size = 0;
				else if(i<PlayerIndex && (Objects[i].Status&0x01))
				{
					Size = 13;
				}
				else if(i<PlayerIndex && (Objects[i].Status&0x02))
					Size = 20;
				else if(i<PlayerIndex && (Objects[i].Status&0x04))
					Size = 30;
				else if(i>SaucerIndex)
					Size = 4;
				else
					assert(0);
				int x = int(Objects[i].PositionX) * cx / 0x2000;
				int y = int(Objects[i].PositionY) * cy / 0x1800;
				y = cy - y;
				if(Size)
					Ellipse(hdc, x-Size*cx/1033, y-Size*cy/733, x+Size*cx/1033, y+Size*cy/733);
				else
					Rectangle(hdc, x-10, y-10, x+10, y+10);
				if(i==PlayerIndex)
				{
					MoveToEx(hdc, x, y, 0);
					LineTo(hdc, int(cos(float(ShipDirection)/255.f*6.283f)*20.0f)+x,
								int(sin(float(ShipDirection)/255.f*6.283f)*-20.0f)+y);
				}
			}
			SelectObject(hdc, GetStockObject(WHITE_BRUSH));
		}

		for(int i = 0; i < 2; i++)
		{
			if(BestDirInfo[i].CollisionAtTime != MaxInt)
			{
				int x = int(BestDirInfo[i].StartX) * cx / 0x2000;
				int y = int(BestDirInfo[i].StartY) * cy / 0x1800;
				y = cy - y;
				MoveToEx(hdc, x, y, 0);
				x = BestDirInfo[i].PositionXEasy(BestDirInfo[i].ShotTime) * cx / 0x2000;
				y = BestDirInfo[i].PositionYEasy(BestDirInfo[i].ShotTime) * cy / 0x1800;
				y = cy - y;
				LineTo(hdc, x, y);
			}
			SelectObject(hdc, GetStockObject(WHITE_PEN));
		}
		
		/*for(int x = 0; x < cx; x++)
		for(int y = 0; y < cy; y++)
		{
			int xAst = x * 0x2000 / cx;
			int yAst = (cy-y) * 0x1800 / cy;
			for(int Object = SaucerIndex; Object >= 0; Object--)
			{
				if(char(VarO(RAMO_Status, Object)) <= 0)
					continue;
				bool collision = AreObjectsNear(0x1F, xAst, yAst,
												Object,
												MakeWord(VarO(RAMO_PositionX, Object), VarO(RAMO_PositionExactX, Object)),
												MakeWord(VarO(RAMO_PositionY, Object), VarO(RAMO_PositionExactY, Object)));
				if(collision)
					SetPixel(hdc, x, y, RGB(255,0,0));
			}
		}*/

		BitBlt(hdcScreen, 0, 0, cx, cy, hdc, 0, 0, SRCCOPY);
		DeleteDC(hdc);
		DeleteObject(hBitmap);
	}
	bool Check(const byte* ram, const KeysPacket& keys, const ShotInfo& ShotInfo, bool set)
	{
/*		byte s1 = Var(RAM_RandomSeed1);
		byte s2 = Var(RAM_RandomSeed2);
		Rand();
		int i = 0;
		while(s1 != Var(RAM_RandomSeed1) || s2 != Var(RAM_RandomSeed2))
		{
			printf("%i: %x%x\n", ++i, Var(RAM_RandomSeed1), Var(RAM_RandomSeed2));
			Rand();
		}
		while(true);*/

		struct
		{
			void* pVariable;
			int Address;
		}
		VariableAddresses[] =
		{
			&CurrentPlayer,            0x18,
			&NumPlayers,			   0x1C,
			&PlayerScoreTens,		   0x52,
			&PlayerScoreThousends,	   0x53,
			&cShips,				   0x57,
			&HyperSpaceStatus,		   0x59,
			&FastTimer,				   0x5C,
			&SlowTimer,				   0x5D,
			&RandomSeed1,			   0x5F,
			&RandomSeed2,			   0x60,
			&ShipDirection,			   0x61,
			&FireHyperspaceToggle,	   0x63,
			&AccX,					   0x64,
			&AccY,					   0x65,
			&NumAsteroidsStart,        0x02F5,
			&NumAsteroids,			   0x02F6,
			&SaucerCountdown,		   0x02F7,
			&SaucerCountdownStart,	   0x02F8,
			&InvisibleTimer,		   0x02FA,
			&NewLevelStartCountdown,   0x02FB
		};

		if(set)
		{
			for(int i = 0; i < _countof(VariableAddresses); i++)
				*((byte*)VariableAddresses[i].pVariable) = ram[VariableAddresses[i].Address];
			for(int i = 0; i < NumObjects; i++)
			{
				Objects[i].Status = char(ram[0x200+i]);
				Objects[i].VelocityX = char(ram[0x223+i]);
				Objects[i].VelocityY = char(ram[0x246+i]);
				Objects[i].PositionX = MakeWord(ram[0x269+i], ram[0x2AF+i]);
				Objects[i].PositionY = MakeWord(ram[0x28C+i], ram[0x2D2+i]);
			}
			return false;
		}

	
		bool errors = false;
		for(int i = 0; i < _countof(VariableAddresses); i++)
			if(//i!=RAM_RandomSeed1 && i!=RAM_RandomSeed2 &&
				//i!=RAM_FireHyperspaceToggle &&
				VariableAddresses[i].pVariable!=&cShips &&
				//i!=RAM_PlayerScoreTens && i!=RAM_PlayerScoreThousends &&
				VariableAddresses[i].pVariable!=&SaucerCountdown && VariableAddresses[i].pVariable!=&SaucerCountdownStart &&
				//i!=RAM_PositionExactX && i!=RAM_PositionExactY &&
				//i!=RAM_PositionX && i!=RAM_PositionY &&
				//i!=RAM_VelocityY && i!=RAM_AccY &&
				*((byte*)VariableAddresses[i].pVariable) != ram[VariableAddresses[i].Address])
			{
				printf("Byte an Adresse %x verschieden: berechnet: %x tatschlich %x\n", VariableAddresses[i].Address, *((byte*)VariableAddresses[i].pVariable), ram[VariableAddresses[i].Address]);
				errors = true;
			}

		for(int i = 0; i < NumObjects; i++)
		{
			if( Objects[i].Status != char(ram[0x200+i]) ||
				Objects[i].VelocityX != char(ram[0x223+i]) ||
				Objects[i].VelocityY != char(ram[0x246+i]) ||
				Objects[i].PositionX != MakeWord(ram[0x269+i], ram[0x2AF+i]) ||
				Objects[i].PositionY != MakeWord(ram[0x28C+i], ram[0x2D2+i]))
			{
				printf("Byte bei Obj %x an Adresse %x verschieden: berechnet: %x tatschlich %x\n",
					i, 0,0,0/*RAMO_Address[j], ram[RAMO_Address[j]+i], ram2[RAMO_Address[j]+i]*/);
				errors = true;
			}
		}

		while(errors)
		{
			*this = last_ram;
			Move(keys, ShotInfo);
		}


		return errors;
	}
	static GameRam last_ram;
	void Move(const KeysPacket& keys, const ShotInfo& ShotInfo)
	{
		last_ram = *this;
		
		if(++FastTimer == 0)
			SlowTimer++;

		//sub_6885
		if(NumPlayers &&
			Objects[PlayerIndex].Status==0 &&
			InvisibleTimer == 0x80)
		{
			InvisibleTimer = 0x10;
		}

		if(!Shoot(keys, ShotInfo))
			assert(ShotInfo.ShootTime != 0);
		DoHyperSpace(keys);
		UpdateHyperspace(keys); //Ruft UpdateDirectionAndThrust() auf
		//UpdateSaucer();

		//Objekte aktualisieren
		for(int i = NumObjects-1; i >= 0; i--)
			MoveObject(i);

		// Kollision fr Spielerschsse(4), Saucerschsse(2), Saucer und Schiff prfen
		for(int i = PlayerShotMaxIndex; i >= PlayerIndex; i--)
			CheckCollisions(i);

		Rand();

		//sub_7BC0

		byte a = NewLevelStartCountdown;
		if(a)
			NewLevelStartCountdown--;
		if((a | NumAsteroids) == 0)
			ResetAsteroids();
	}
/*	byte Hash()
	{
		byte h = 0;
		for(int i = 0x200; i < 0x2F4; i++)
			h += ram[i];
		h += ram[0x5F];
		h += ram[0x60];
		return h;
	}*/

	bool Shoot(const KeysPacket& keys, const ShotInfo& ShotInfo)//sub_6CD7
	{
		if(NumPlayers == 0)
			return false;
		FireHyperspaceToggle >>= 1;
		if(keys.keys & KeysPacket::KEY_FIRE)
			FireHyperspaceToggle |= 0x80;
		if(!(keys.keys & KeysPacket::KEY_FIRE))
			return false;
		if(FireHyperspaceToggle & 0x40)//letzten Frame schon gedrckt?
			return false;
		if(InvisibleTimer)
			return false;

		int NewShot = PlayerShotMaxIndex;
		while(Objects[NewShot].Status)
			if(NewShot-- == PlayerShotMinIndex)
				return false;

		assert(ShotInfo.Index == NewShot);
		assert(ShotInfo.ShootTime == 0);
		assert(ShotInfo.StartTime == FastTimer);
		ShotInfos[NewShot-PlayerShotMinIndex] = ShotInfo;

		Objects[NewShot] = Objects[PlayerIndex];
		Objects[NewShot].Status = ShotLifetime/4;

		ShootLocationFromAngle(ShipDirection, Objects[NewShot].PositionX, Objects[NewShot].PositionY, Objects[NewShot].VelocityX, Objects[NewShot].VelocityY);
		assert(ShotInfo.StartX == Objects[NewShot].PositionX);
		assert(ShotInfo.StartY == Objects[NewShot].PositionY);
		assert(ShotInfo.VelocityX == Objects[NewShot].VelocityX);
		assert(ShotInfo.VelocityY == Objects[NewShot].VelocityY);
		return true;

/*		//xVelocity
		byte a = Var(RAM_ShipDirection);
		CosLookup(a);
		bool c = a >= 0x80;
		RotateRight(a, c);
		byte tmp9 = a;
		a += VarO(RAMO_VelocityX, PlayerIndex);
		Clamp(a, 0x91, 0x6F);
		VarO(RAMO_VelocityX, PlayerIndex+NewShot) = a;

		//yVelocity
		a = Var(RAM_ShipDirection);
		SinLookup(a);
		c = a >= 0x80;
		RotateRight(a, c);
		byte tmpC = a;
		a += VarO(RAMO_VelocityY, PlayerIndex);
		Clamp(a, 0x91, 0x6F);
		VarO(RAMO_VelocityY, PlayerIndex+NewShot) = a;

		//xPosition
		a = tmp9;
		byte tmp8 = char(a)<0 ? 0xFF : 0;
		c = a >= 0x80;
		RotateRight(a, c);
		a += tmp9;
		c = false;
		Add(a, VarO(RAMO_PositionExactX, PlayerIndex), c);
		VarO(RAMO_PositionExactX, PlayerIndex+NewShot) = a;
		Add(tmp8, VarO(RAMO_PositionX, PlayerIndex), c);
		VarO(RAMO_PositionX, PlayerIndex+NewShot) = tmp8;

		//yPosition
		a = tmpC;
		byte tmpB = char(a)<0 ? 0xFF : 0;
		c = a >= 0x80;
		RotateRight(a, c);
		a += tmpC;
		c = false;
		Add(a, VarO(RAMO_PositionExactY, PlayerIndex), c);
		VarO(RAMO_PositionExactY, PlayerIndex+NewShot) = a;
		Add(tmpB, VarO(RAMO_PositionY, PlayerIndex), c);
		VarO(RAMO_PositionY, PlayerIndex+NewShot) = tmpB;*/
	}
	static void ShootLocationFromAngle(byte Direction, word& PositionX, word& PositionY, char& VelocityX, char& VelocityY)
	{
		//xVelocity
		byte a = Direction;
		CosLookup(a);
		bool c = a >= 0x80;
		RotateRight(a, c);
		byte tmp9 = a;
		a += byte(VelocityX);
		Clamp(a, 0x91, 0x6F);
		VelocityX = char(a);

		//yVelocity
		a = Direction;
		SinLookup(a);
		c = a >= 0x80;
		RotateRight(a, c);
		byte tmpC = a;
		a += byte(VelocityY);
		Clamp(a, 0x91, 0x6F);
		VelocityY = char(a);

		//xPosition
		/*a = tmp9;
		byte tmp8 = char(a)<0 ? 0xFF : 0;
		c = a >= 0x80;
		RotateRight(a, c);
		a += tmp9;
		c = false;
		Add(a, LowByte(PositionX), c);
		LowByte(PositionX) = a;
		Add(tmp8, HighByte(PositionX), c);
		HighByte(PositionX) = tmp8;*/
		PositionX += char(tmp9) + char(Half(tmp9));

		//yPosition
/*		a = tmpC;
		byte tmpB = char(a)<0 ? 0xFF : 0;
		c = a >= 0x80;
		RotateRight(a, c);
		a += tmpC;
		c = false;
		Add(a, LowByte(PositionY), c);
		LowByte(PositionY) = a;
		Add(tmpB, HighByte(PositionY), c);
		HighByte(PositionY) = tmpB;*/
		PositionY += char(tmpC) + char(Half(tmpC));

	}

	void DoHyperSpace(const KeysPacket& keys)//sub_6E74
	{
		if(!(keys.keys & KeysPacket::KEY_HYPERSPACE) ||
			NumPlayers == 0 ||
			InvisibleTimer ||
			Objects[PlayerIndex].Status < 0)
			return;

		Objects[PlayerIndex].Status = 0;
		Objects[PlayerIndex].VelocityX = 0;
		Objects[PlayerIndex].VelocityY = 0;
		InvisibleTimer = 0x30;

		byte a = Rand() & 0x1F;
		if(a >= 0x1D)
			a = 0x1C;
		if(a < 3)
			a = 3;
		HighByte(Objects[PlayerIndex].PositionX) = a;
	
		for(int i = 0; i < 4; i++)
			Rand();
		a = Rand() & 0x1F;
		HyperSpaceStatus = 1;
		if(a >= 0x18)
		{
			a &= 7;
			bool c;
			ShiftLeft(a, c);
			Add(a, 4, c);
			if(a >= NumAsteroids)//HyperSpace fehlgeschlagen
				HyperSpaceStatus = -128;
		}
		Clamp(a, 3, 0x14);
		HighByte(Objects[PlayerIndex].PositionY) = a;
	}

	void UpdateHyperspace(const KeysPacket& keys)//sub_703F
	{
		if(NumPlayers == 0 ||
			Objects[PlayerIndex].Status < 0)
			return;
		if(InvisibleTimer == 0)
		{
			UpdateDirectionAndThrust(keys);
			return;
		}
		if(--InvisibleTimer)
			return;
		if(char(HyperSpaceStatus) < 0) goto loc_706F;
		if(HyperSpaceStatus) goto loc_7068;
		if(!CanShipAppear()) goto loc_7081;
		if(Objects[SaucerIndex].Status)
		{
			InvisibleTimer = 2;
			return;
		}
loc_7068:
		Objects[PlayerIndex].Status = 1;
		goto loc_7081;
loc_706F:
		Objects[PlayerIndex].Status = -96;
		cShips--;
		InvisibleTimer = 0x81;
loc_7081:
		HyperSpaceStatus = 0;
	}

	bool CanShipAppear()
	{
		for(int i = SaucerIndex; i >= 0; i--)
		{
			if(Objects[i].Status == 0)
				continue;

			byte dif = HighByte(Objects[i].PositionX)-HighByte(Objects[PlayerIndex].PositionX);
			if(dif < 0xFC && dif >= 4)
				continue;

			dif = HighByte(Objects[i].PositionY)-HighByte(Objects[PlayerIndex].PositionY);
			if(dif < 0xFC && dif >= 4)
				continue;

			InvisibleTimer++;
			return false;
		}
		return true;
	}

	void UpdateDirectionAndThrust(const KeysPacket& keys)//loc_7086
	{
		if(keys.keys & KeysPacket::KEY_LEFT)
			ShipDirection += RotationSpeed;
		else if(keys.keys & KeysPacket::KEY_RIGHT)
			ShipDirection -= RotationSpeed;
		
		if(FastTimer & 1)
			return;
		if(keys.keys & KeysPacket::KEY_THRUST)
		{
			byte y = 0;
			byte a = ShipDirection;//a=0
			CosLookup(a);
			if(char(a) < 0)
				y--;
			a <<= 1;   //a=80/FE
			bool c = false;
			Add(a, AccX, c);//a=f9  c=1
			byte x = a;//x=77
			a = y;
			Add(a, Objects[PlayerIndex].VelocityX, c);//a=2
			sub7125(a, x);
			Objects[PlayerIndex].VelocityX = a;
			AccX = x;

			y = 0;
			a = ShipDirection;
			SinLookup(a);
			if(char(a) < 0)
				y--;
			a <<= 1;
			c = false;
			Add(a, AccY, c);
			x = a;
			a = y;
			Add(a, Objects[PlayerIndex].VelocityY, c);
			sub7125(a, x);
			Objects[PlayerIndex].VelocityY = a;
			AccY = x;
		}
		else
		{
			// KEY_THRUST nicht gedrckt:
			if(Objects[PlayerIndex].VelocityX | AccX)
			{
				byte a = (Objects[PlayerIndex].VelocityX << 1) ^ 0xFF;
				bool c = char(a) >= 0;
				byte x = c ? 0 : 0xFF;
				Add(AccX, a, c);
				Add((byte&)Objects[PlayerIndex].VelocityX, x, c);
			}
			if(Objects[PlayerIndex].VelocityY | AccY)
			{
				byte a = (Objects[PlayerIndex].VelocityY << 1) ^ 0xFF;
				bool c = char(a) >= 0;
				byte x = c ? 0 : 0xFF;
				Add(AccY, a, c);
				Add((byte&)Objects[PlayerIndex].VelocityY, x, c);
/*		
				char a = Var(RAM_VelocityY)<<1;
				char x = 0xFF;
				a ^= 0xFF;
				bool c = false;
				if(a >= 0)
				{
					x++;
					c = true;
				}
				int sum = int(a)+int(Var(RAM_AccY))+c;
				bool overflow = (sum & 0xFF00) != 0;
				a = sum;
				Var(RAM_AccY) = a;
				a = x;
				a += Var(RAM_VelocityY) + overflow;
				Var(RAM_VelocityY) = a;*/
			}
		}
	}


	void MoveObject(int i)
	{
		//byte tmp4, tmp5, tmp6, tmp7;
		byte a = Objects[i].Status;
		if(a == 0)
			return;
		//loc_6F62
		if(char(a) < 0)
		{
			//Objekt explodiert
			a = ((a^0xFF)+1) >> 4;
			bool c = true;
			if(i == PlayerIndex)
			{
				c = (FastTimer&1) != 0;
				a = 0;
			}
			//loc_6F77
			Add(a, Objects[i].Status, c);
			if(char(a) < 0)
			{//loc_6FA1
				Objects[i].Status = a;
				//a = (a&0xF0)+0x10;
				//if(i == PlayerIndex)
				//	a = 0;
				//byte y = a;
				//4567a = VarO(RAMO_PositionExactX, i);
				{//loc_7027
					//sub_72FE
				}
			}
			else if(i == PlayerIndex)
			{//loc_6F93
				ResetPlayerPosition();
				Objects[i].Status = 0;
			}
			else if(i > PlayerIndex)
			{//loc_6F99
			}
			else
			{
				if(--NumAsteroids == 0)
					NewLevelStartCountdown = 0x7F;
				Objects[i].Status = 0;
			}
			return;
		}
		//loc_6FC7
//		byte y = 0;

		//xPosition += char(VarO(RAMO_VelocityX, i));
		/*a = VarO(RAMO_VelocityX, i);
		if(char(a) < 0)
			y--;
		bool c = false;
		Add(a, VarO(RAMO_PositionExactX, i), c);
		VarO(RAMO_PositionExactX, i) = a;
		a = y;
		Add(a, VarO(RAMO_PositionX, i), c);*/
//		if(a >= 0x20)//Objekt geht rechts raus
		if(MoveLinearXSingle(Objects[i].PositionX, Objects[i].VelocityX))
		{
			if(i == SaucerIndex)
			{
				//sub_702D
				SaucerCountdown = SaucerCountdownStart;
				Objects[SaucerIndex].Status = 0;
				Objects[SaucerIndex].VelocityX = 0;
				Objects[SaucerIndex].VelocityY = 0;
				return;
			}
		}
//		MakeBytesFromWord(xPosition, VarO(RAMO_PositionX, i), VarO(RAMO_PositionExactX, i));
		//loc_6FEC
		//VarO(RAMO_PositionX, i) = a;

		Objects[i].PositionY = MoveLinearY(Objects[i].PositionY, Objects[i].VelocityY, 1);
//		MakeBytesFromWord(yPosition, VarO(RAMO_PositionY, i), VarO(RAMO_PositionExactY, i));

/*		y = 0;
		a = VarO(RAMO_VelocityY, i);
		if(char(a) < 0)
			y--;
		bool c = false;
		Add(a, VarO(RAMO_PositionExactY, i), c);
		VarO(RAMO_PositionExactY, i) = a;
		a = y;
		Add(a, VarO(RAMO_PositionY, i), c);
		if(a >= 0x18)
		{
			if(a == 0x18)
			{//loc_7011
				a = 0;
			}
			else
				a = 0x17;
		}
		//loc_7013
		VarO(RAMO_PositionY, i) = a;*/

		/*a = Objects[i].Status;
		y = 0xE0;
		if((a&1) == 0)
		{
			y = 0xF0;
			if((a&2) == 0)
				y = 0;
		}*/
		sub_72FE(i);
	}

	void sub_72FE(int i)
	{
	/*	byte tmp0 = y;
		bool c = false;
		a = tmp5;
		ShiftRight(a, c);
		RollRight(tmp4, c);
		ShiftRight(a, c);
		RollRight(tmp4, c);
		ShiftRight(a, c);
		RollRight(tmp4, c);
		tmp5 = a;

		a = tmp7+4;
		c = false;
		ShiftRight(a, c);
		RollRight(tmp6, c);
		ShiftRight(a, c);
		RollRight(tmp6, c);
		ShiftRight(a, c);
		RollRight(tmp6, c);
		tmp7 = a;
		byte x = 4;
		//sub_7C1C
		a = 0x70;
		a -= tmp0;
		while(char(a) >= -96)
		{
			byte stack = a;
			a = 0x90;
			//sub_7CDE
			a = stack;
			a -= 0x10;
		}
		//loc_733B
		//sub_7CDE
		a = Objects[i].Status;
		if(char(a) >= 0)
		{//loc_735B
		}
		if(i != PlayerIndex)
		{
			y = (a&0x0C)>>1;
//				a = 50F8[y];
//				x = 50F9[y];
			if(x != 0)
			{
				//loc_7370
			}
		}
		{//sub_7465
			
		}*/
//loc_7384:
		//...
		if(i>SaucerIndex && (FastTimer&3) == 0)
		{
			Objects[i].Status--;
			if(!updating && !Objects[i].Status)
			{//////////
				printf("Shot %i ins Leere gegangen um %x\n", i-PlayerShotMinIndex, FastTimer);
				ErrorsOccured = true;
			}
		}
	}

	void ResetAsteroids();


	void ResetPlayerPosition()//sub_71E8
	{
		Objects[PlayerIndex].PositionX = 0x1060;
		Objects[PlayerIndex].PositionY = 0x0C60;

		Objects[PlayerIndex].VelocityX = 0;
		Objects[PlayerIndex].VelocityY = 0;
	}

	static void CosLookup(byte& a)//sub_77D2
	{
		a += 0x40;
		SinLookup(a);
	}

	static void SinLookup(byte& a)//sub_77D5
	{
		const byte Table[0x41] =//0x57B9
		{
			0x00, 0x03, 0x06, 0x09, 0x0c, 0x10, 0x13, 0x16, 0x19, 0x1c, 0x1f, 0x22, 0x25, 0x28,
			0x2b, 0x2e, 0x31, 0x33, 0x36, 0x39, 0x3c, 0x3f, 0x41, 0x44, 0x47, 0x49, 0x4c, 0x4e,
			0x51, 0x53, 0x55, 0x58, 0x5a, 0x5c, 0x5e, 0x60, 0x62, 0x64, 0x66, 0x68, 0x6a, 0x6b,
			0x6d, 0x6f, 0x70, 0x71, 0x73, 0x74, 0x75, 0x76, 0x78, 0x79, 0x7a, 0x7a, 0x7b, 0x7c,
			0x7d, 0x7d, 0x7e, 0x7e, 0x7e, 0x7f, 0x7f, 0x7f, 0x7f
		};
		bool inv = false;
		if(char(a) < 0)
		{
			inv = true;
			a &= 0x7F;
		}
		if(a >= 0x41)
			a = (a^0x7F) + 1;
		a = Table[a];
		if(inv)
			a = (a^0xFF) + 1;
	}

	void sub7125(byte& a, byte& x)
	{
		if(char(a) >= 0)
		{
			if(a >= 0x40)
			{
				x = 0xFF;
				a = 0x3F;
			}
		}
		else
		{
			if(a < 0xC0)
			{
				x = 1;
				a = 0xC0;
			}
		}
	}

	void CloneAsteroid(byte Dest, byte Src)//sub_6A9D
	{
		Objects[Dest] = Objects[Src];
		Objects[Dest].Status = (Objects[Src].Status&7) | (Rand()&0x18);
	}

	void SplitAsteroid(byte CollisionCauser, byte Asteroid)//Erzeugt zwei kleine Asteroiden
	{//sub_75EC
		//byte a = Objects[y].Status;
		//a &= 0x78;
		//byte tmp_e = a;
		byte NewStatus = (Objects[Asteroid].Status&7) >> 1;
		if(NumPlayers && (CollisionCauser == PlayerIndex || CollisionCauser >= PlayerShotMinIndex))
		{
			//Der Schuss ist vom Spieler oder der Spieler kollidiert selber mit dem Asteroid
			if(NewStatus == 0)
				AddScore(10);
			else if(NewStatus == 1)
				AddScore(5);
			else if(NewStatus == 2)
				AddScore(2);
			else
				assert(0);
		}
		if(NewStatus)
			Objects[Asteroid].Status = (NewStatus | (Objects[Asteroid].Status&0x78));
		else
		{
			Objects[Asteroid].Status = 0;
			return;
		}
		byte NewAsteroid = NextAsteroidSlot(PlayerIndex-1);
		if(NewAsteroid == 0xFF)
		{//////////
			//MessageBeep(0);
			printf("Keine Slots mehr uebrig!!\n");
			return;
		}
		NumAsteroids++;
		CloneAsteroid(NewAsteroid, Asteroid);
		sub_7203(NewAsteroid, Asteroid);
		LowByte(Objects[NewAsteroid].PositionX) ^= (Objects[NewAsteroid].VelocityX&0x1F) << 1;
		NewAsteroid = NextAsteroidSlot(NewAsteroid-1);
		if(NewAsteroid == 0xFF)
		{
			//MessageBeep(0);
			printf("Nur noch ein Slot uebrig!!\n");
			return;
		}
		NumAsteroids++;
		CloneAsteroid(NewAsteroid, Asteroid);
		sub_7203(NewAsteroid, Asteroid);
		LowByte(Objects[NewAsteroid].PositionY) ^= (Objects[NewAsteroid].VelocityY&0x1F) << 1;
	}

	byte NextAsteroidSlot(byte StartSearchAt)
	{//sub_745A
		byte i = StartSearchAt;
		while(i!=0xFF && Objects[i].Status)
			i--;
		return i;
	}

	void sub_7203(byte& x, byte& y)
	{
		byte a = Rand() & 0x8F;
		if(char(a) < 0)
			a |= 0xF0;
		a += Objects[y].VelocityX;
		sub_7233(a);
		Objects[x].VelocityX = a;
	
		Rand(); Rand(); Rand();

		a = Rand() & 0x8F;
		if(char(a) < 0)
			a |= 0xF0;
		a += Objects[y].VelocityY;
		sub_7233(a);
		Objects[x].VelocityY = a;
	}

	void sub_7233(byte& a)
	{
		if(char(a) < 0)
		{
			if(a < 0xE1)
				a = 0xE1;
			else if(a >= 0xFB)
				a = 0xFA;
		}
		else
		{
			if(a < 6)
				a = 6;
			else if(a >= 0x20)
				a = 0x1F;
		}
	}

	void AddScore(byte ToAdd)
	{//sub_7397
		word Score = DecimalToBin(PlayerScoreThousends, PlayerScoreTens);
		if(Score/1000 != (Score+ToAdd)/1000)
			cShips++;
		Score += ToAdd;
		if(pScore)
			*pScore += ToAdd;
		BinToDecimal(Score, PlayerScoreThousends, PlayerScoreTens);
	}
	void CheckCollisions(byte CollisionCauser);

	static bool AreObjectsNear(byte i, word iPositionX, word iPositionY, byte iStatus,
							   byte j, word jPositionX, word jPositionY, byte jStatus)
	{//Ship/Saucer/Shots muss in i sein, Asteroiden in j
		byte a = LowByte(jPositionX);
		bool c = true;
		Subtract(a, LowByte(iPositionX), c);
		byte tmp8 = a;
		a = HighByte(jPositionX);
		Subtract(a, HighByte(iPositionX), c);
		ShiftRight(a, c);
		RotateRight(tmp8, c);
		ShiftLeft(a, c);
		if(a)
		{
			if(char(a) >= 0)
				return false;
			a ^= 0xFE;
			if(a)
				return false;
			tmp8 ^= 0xFF;
		}
		a = LowByte(jPositionY);
		c = true;
		Subtract(a, LowByte(iPositionY), c);
		byte tmp9 = a;
		a = HighByte(jPositionY);
		Subtract(a, HighByte(iPositionY), c);
		ShiftRight(a, c);
		RotateRight(tmp9, c);
		ShiftLeft(a, c);
		if(a)
		{
			if(char(a) >= 0)
				return false;
			a ^= 0xFE;
			if(a)
				return false;
			tmp9 ^= 0xFF;
		}
		a = 0x2A;
		ShiftRight(jStatus, c);
		if(!c)
		{
			a = 0x48;
			ShiftRight(jStatus, c);
			if(!c)
				a = 0x84;
		}
		if(i == PlayerIndex)
			a += 0x1C;
		else if(i == SaucerIndex)
		{
			c = true;
			Add(a, 0x12, c);
			if(iStatus != 1)
				Add(a, 0x12, c);
		}
		if(a < tmp8) return false;
		if(a < tmp9) return false;
		byte tmp = a;
		a >>= 1;
		c = false;
		Add(a, tmp, c);
		tmp = a;
		a = tmp9;
		Add(a, tmp8, c);
		if(c) return false;
		if(a >= tmp) return false;
		return true;
	}

	// Gibt zurck, ob sich die Asteroiden oder das UFO gendert haben
	bool OnCollision(byte CollisionCauser, byte& y)//sub_6B0F
	{
		if(CollisionCauser != SaucerIndex)
		{
			if(CollisionCauser != PlayerIndex)
			{
				// CollisionCauser ist ein Schuss
				Objects[CollisionCauser].Status = 0; //Schuss lschen
				if(y == PlayerIndex)
				{
					cShips--;
					InvisibleTimer = 0x81;
				}
				else if(y > PlayerIndex)
					DestroySaucer();
				else
					SplitAsteroid(CollisionCauser, y);
				Objects[y].Status = -96;
				Objects[y].VelocityX = 0;
				Objects[y].VelocityY = 0;
				return y != PlayerIndex;
			}
			InvisibleTimer = 0x81;
			cShips--;
			CollisionCauser = PlayerIndex;
		}
		else if(y == PlayerIndex)
		{
			CollisionCauser = PlayerIndex;
			y = SaucerIndex;
			InvisibleTimer = 0x81;
			cShips--;
		}
		Objects[CollisionCauser].Status = -96;
		Objects[CollisionCauser].VelocityX = 0;
		Objects[CollisionCauser].VelocityY = 0;
		if(y < PlayerIndex)
		{
			SplitAsteroid(CollisionCauser, y);
		}
		else
		{
			DestroySaucer();
		}
		Objects[y].Status = -96;
		Objects[y].VelocityX = 0;
		Objects[y].VelocityY = 0;
		return true;
	}

	void DestroySaucer()
	{
		SaucerCountdown = SaucerCountdownStart;
		if(NumPlayers)
		{
			if(Objects[SaucerIndex].Status == 1)
				AddScore(99);
			else
				AddScore(20);
		}
	}

	byte Rand()
	{
		bool carry = false;
		ShiftLeft(RandomSeed1, carry);
		if(char(RotateLeft(RandomSeed2, carry)) < 0)
			RandomSeed1++;
		if(RandomSeed1 & 2)
			RandomSeed1 ^= 1;
		if((RandomSeed1|RandomSeed2) == 0)
			RandomSeed1++;
		return RandomSeed1;
	}
};

struct FramePacket
{
	char vectorram[1024];
	char frameno;  // wird bei jedem Frame inkrementiert
	char ping;     // Der Server schickt das letzte empfangene ping-byte zurck
	byte ram[0x300];
};

#pragma pack()


class Player
{
public:
	Player(SOCKET sd, ADDRESS server_ip) : sd(sd), server_ip(server_ip) {};
	void Run(void);
	void InterpretScreen(FramePacket &packet, GameStatus& game);
	void ReceivePacket(FramePacket &packet);
	void SendPacket(KeysPacket &packet);
private:
	SOCKET sd;
	ADDRESS server_ip;
};
